package org.hathitrust.lrl.core;

import java.io.BufferedReader;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;

import javax.net.ssl.HttpsURLConnection;

import org.json.JSONArray;
import org.json.JSONObject;

public class UtilIO {
	public enum CaseProcessing { AsIs, ToLower }
	
	// LRL = Low-Resource Language
    public final static String LRL_HOME;
    
    static {
    	String lrl_home = System.getenv("LRL_HOME");

    	if ((lrl_home == null) || (lrl_home.equals(""))) {
    		String cwd = System.getProperty("user.dir"); // current working diretory

    		System.err.println("Warning: Environment variable LRL_HOME not set");		
    		System.err.println("  Defaulting to: " + cwd);

    		LRL_HOME=cwd;
    	}
    	else {
    		LRL_HOME=lrl_home;
    	}
    }
    
	public static ArrayList<String> readLineBasedTextFile(String input_filename, CaseProcessing case_mode) 
	{
		 ArrayList<String> text_lines = new ArrayList<String>();
		
		try {
			File fileDir = new File(input_filename);

			FileInputStream fis = new FileInputStream(fileDir);
			InputStreamReader isr = new InputStreamReader(fis, "UTF8");
			BufferedReader brin = new BufferedReader(isr);
		
			String intput_line;
			while ((intput_line = brin.readLine()) != null) {
				if (case_mode == CaseProcessing.ToLower) {
					intput_line = intput_line.toLowerCase();
				}
				text_lines.add(intput_line);
			}

			brin.close();
		} 
		catch (Exception e)  {
			e.printStackTrace();
		}
		
		return text_lines;
	}
	
	public static ArrayList<String> readLineBasedTextFile(String input_filename) 
	{
		return readLineBasedTextFile(input_filename, CaseProcessing.AsIs);
	}
	
	public static String readTextFile(String input_filename)
	{
		ArrayList<String> text_lines = readLineBasedTextFile(input_filename);
		
		String text = String.join("\n", text_lines);
		
		return text;
	}
			
	public static JSONObject readJSONFile(String input_filename) 
	{
		JSONObject json_obj = null;
		
		String json_str = readTextFile(input_filename);
		try {
			json_obj = new JSONObject(json_str);
		}
		catch (Exception e) {
			e.printStackTrace();
		}
		
		return json_obj;
	}

	public static String getFromURL(String url)
	{
        StringBuffer text_buffer = new StringBuffer();

        try {
                URL get_url = new URL(url);
                HttpsURLConnection conn = (HttpsURLConnection)get_url.openConnection();

                InputStream is = conn.getInputStream();
                InputStreamReader isr = new InputStreamReader(is,StandardCharsets.UTF_8);
                BufferedReader reader = new BufferedReader(isr);

                String input_line = null;
                while ((input_line = reader.readLine()) != null) {
                	text_buffer.append(input_line);
                      
                }
                
                reader.close();
                
        } 
        catch (Exception e) {
                e.printStackTrace();
            
        }

        String text = text_buffer.toString();
        return text;
	}
	
	
	public static void downloadEFJson(String id, String output_dir )
	{
		// Example of full URL
		//   https://solr2.htrc.illinois.edu/htrc-ef-access/get?action=download-ids&ids=uc1.31822003178597-seq-000044&output=json		 
	
		// First check to see if we need to download it at all
		
		String id_filesafe = id.replaceAll("/", "%2F").replaceAll(":", "%3A");
		String output_filename = output_dir + File.separator + id_filesafe + ".json";
		File output_file = new File(output_filename);
		if (output_file.exists() && (output_file.length()>0)) {
			System.out.println("  Skipping as downloaded JSON file already exits: ");
			System.out.println("    " + output_filename);
			return;
		}
		
		// Get to here, download via HTRC Access API
		String base_url = "https://solr2.htrc.illinois.edu/htrc-ef-access/get";
		
		String args = "action=download-ids";
		args += "&ids=" + id;
		args += "&output=json";
		
		String get_url = base_url + "?" + args;
		
		String json_ef_page_str = UtilIO.getFromURL(get_url);
		
		UtilIO.writeTextToFile(output_filename, json_ef_page_str);
	}
	
	public static String postQueryToURL(String url, String post_url_params)
	{
        StringBuffer text_buffer = new StringBuffer();
		 
        byte[] post_data = post_url_params.getBytes(StandardCharsets.UTF_8);
        int post_data_len = post_data.length;

        try {
                URL post_url = new URL(url);
                //HttpURLConnection conn = (HttpURLConnection) post_url.openConnection();
                HttpsURLConnection conn = (HttpsURLConnection)post_url.openConnection();
                
                conn.setDoOutput(true);
                conn.setInstanceFollowRedirects(false);
                conn.setRequestMethod("POST");
                conn.setRequestProperty("Content-Type", "application/x-www-form-urlencoded");
                conn.setRequestProperty("charset", "utf-8");
                conn.setRequestProperty("Content-Length", Integer.toString(post_data_len));
                conn.setUseCaches(false);

                // try-resource syntax auto-closes stream
                try (DataOutputStream dos = new DataOutputStream(conn.getOutputStream())) {
                        dos.write(post_data);
                }
                

                InputStream is = conn.getInputStream();
                InputStreamReader isr = new InputStreamReader(is,StandardCharsets.UTF_8);
                BufferedReader reader = new BufferedReader(isr);

                String input_line = null;
                while ((input_line = reader.readLine()) != null) {
                	text_buffer.append(input_line);
                      
                }
               
        } 
        catch (Exception e) {
                e.printStackTrace();
            
        }

        String text = text_buffer.toString();
        return text;
	}
	
	
	public static ArrayList<String> querySolrEF(String query, int num_rows_per_page)
	{		 
		// https://solr2.htrc.illinois.edu/robust-solr/solr3456-faceted-htrc-full-ef16/select
	
		//   q=(alllangs_htrctokentext:atu) OR (alllangs_htrctokentext:haere)&indent=on
		//   &wt=json&start=0&rows=15
		//   &facet=on&facet.field=volumegenre_htrcstrings&facet.field=volumelanguage_htrcstring&facet.field=volumerightsAttributes_htrcstring&facet.field=volumenames_htrcstrings&facet.field=volumepubPlace_htrcstring&facet.field=volumebibliographicFormat_htrcstring&facet.field=volumeclassification_lcc_htrcstrings&facet.field=volumeconcept_htrcstrings
		
		ArrayList<String> ids = new ArrayList<String>();
		
		String base_url = "https://solr2.htrc.illinois.edu/robust-solr/solr3456-faceted-htrc-full-ef16/select";
		String query_arg =  "q=" + query;
		String extra_args = "wt=json&start=0&rows=" + num_rows_per_page;
		
		String args = query_arg + "&" + extra_args;
		
		System.out.println("  Initiating SolrEF query with args:" + args);
		String json_result_set_str = UtilIO.postQueryToURL(base_url,args);
		
		try {
		  JSONObject json_result_set = new JSONObject(json_result_set_str);
          JSONObject json_result_set_response = json_result_set.getJSONObject("response");
          
          long num_found = json_result_set_response.getLong("numFound");
          
          System.out.println("  Downloading up-to the first " + num_rows_per_page + " of " + num_found + " matching pages" );
          
          
          JSONArray docs = json_result_set_response.getJSONArray("docs");
          
          for (int i=0; i<docs.length(); i++) {
        	  JSONObject json_doc_rec = docs.getJSONObject(i);
        	  String id = json_doc_rec.getString("id");
        	  ids.add(id);
          }
		}
		catch (Exception e) {
			e.printStackTrace();
		}
		
		return ids;
	}
	

	public static void writeTextToFile(String output_filename, String text) 
	{		
		// try-resource syntax auto-closes stream
		try (FileOutputStream fos = new FileOutputStream(output_filename)) {
			OutputStreamWriter osw = new OutputStreamWriter(fos, StandardCharsets.UTF_8);
			osw.write(text);
			osw.close();
		}
		catch (Exception e) {
			e.printStackTrace();
		}
		
		
	}
	
	public static void writeJSONToFile(String output_filename, JSONObject json_obj) 
	{
		String text = json_obj.toString();
				
		writeTextToFile(output_filename,text);
	}
	
}
